﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Xml;
using System.Linq;
using System.ServiceModel;
using VA.PPMS.CRM.Plugins.Data;
using VA.PPMS.CRM.Plugins.Helper;
using VA.PPMS.CRM.Plugins.SamExclusionSearch;
using VA.PPMS.Context;
using System.Collections.Generic;

namespace VA.PPMS.CRM.Plugins
{
    public class ProviderIdentifierCreate : IPlugin
    {
        private const string ProcessDateAttribute = "ppms_validationcompletedate";
        private const string PluginName = "ProviderIdentifierCreate";
        private IOrganizationService _service;
        private ITracingService _tracingService;
        private PpmsServiceSettings _serviceSettings;
        private static string SecureStorageAccountKey = "";

        /// <summary>
        /// Constructor to grab parameters from plugin registration
        ///   the secure parameter is used to handle SPCode and secrets for each environment, the string payload is XML.
        ///     <settings><SPCode>spc</SPCode><StorageAccountKey>sak</StorageAccountKey></settings>
        /// </summary>
        /// <param name="unsecure"></param>
        /// <param name="secure"></param>
        public ProviderIdentifierCreate(string unsecure, string secure)
        {
            //string xml = "<settings><SPCode>spc</SPCode><StorageAccountKey>sak</StorageAccountKey></settings>";
            SecureStorageAccountKey = SettingsHelper.GetValueFromKey(secure, "StorageAccountKey");

        }

        

        /// <summary>
        /// Plug-in entry point, called when criteria are met
        /// </summary>
        /// <param name="serviceProvider"></param>
        public void Execute(IServiceProvider serviceProvider)
        {
            // Tracing service for debugging
            _tracingService = (ITracingService)serviceProvider.GetService(typeof(ITracingService));

            // Get execution context
            IPluginExecutionContext context = (IPluginExecutionContext)serviceProvider.GetService(typeof(IPluginExecutionContext));

            if (context.InputParameters.Contains("Target") && context.InputParameters["Target"] is Entity)
            {
                _tracingService.Trace("Begin");

                // Obtain the target entity from the input parameters.
                Entity target = (Entity)context.InputParameters["Target"];

                // Verify target entity type
                if (target.LogicalName != "ppms_provideridentifier")
                    return;

                _tracingService.Trace("Entity found: {0}", target.Id.ToString());

                // Get organization service reference
                IOrganizationServiceFactory serviceFactory = (IOrganizationServiceFactory)serviceProvider.GetService(typeof(IOrganizationServiceFactory));
                _service = serviceFactory.CreateOrganizationService(context.UserId);

                try
                {
                    // Check for NPI type
                    OptionSetValue providerIdType = target.GetAttributeValue<OptionSetValue>("ppms_identifiertype");

                    _tracingService.Trace("Processing type: {0}", providerIdType.Value);

                    switch (providerIdType.Value)
                    {
                        case (int)PpmsHelper.ProviderIdentifierType.NPI:
                            ValidateNpi(target);
                            break;
                        case (int)PpmsHelper.ProviderIdentifierType.TIN:
                            ValidateTin(target);
                            break;
                        default:
                            _tracingService.Trace("Target entity is not an NPI or TIN.");
                            break;
                    }
                }
                catch (FaultException<OrganizationServiceFault> ex)
                {
                    _tracingService.Trace("Fault: {0}", ex.ToString());
                    throw new InvalidPluginExecutionException(String.Format("An error occurred in {0}.", PluginName), ex);
                }
                catch (Exception ex)
                {
                    _tracingService.Trace("Exception: {0}", ex.ToString());
                    throw;
                }
            }
            _tracingService.Trace("Done");
        }

        /// <summary>
        /// Validate the NPI value LEIE and NPPES lists
        /// </summary>
        /// <param name="target">ProviderIdentifier entity</param>
        private void ValidateNpi(Entity target)
        {
            bool isDeactivateRequested = false;

            OptionSetValue providerNpiType = target.GetAttributeValue<OptionSetValue>("ppms_npitype");
            if (providerNpiType == null)
            {
                _tracingService.Trace("Target entity does not contain Npi Type.");
                return;
            }

            string npiType;
            switch (providerNpiType.Value)
            {
                case (int)PpmsHelper.ProviderNpiType.Individual:
                    npiType = "1";
                    break;
                case (int)PpmsHelper.ProviderNpiType.GroupAgency:
                    npiType = "2";
                    break;
                case (int)PpmsHelper.ProviderNpiType.Organizational:
                    npiType = "2";
                    break;
                default:
                    _tracingService.Trace("Npi Type not found.");
                    return;
            }

            var entity = GetNpi(target);
            if (entity == null)
            {
                _tracingService.Trace("Target entity not found.");
                return;
            }

            // Get associated provider reference
            var provider = entity.GetAttributeValue<EntityReference>("ppms_provider");
            // Get current batch detail
            var detail = GetCurrentBatchDetail(provider);
            var npi = entity.GetAttributeValue<string>("ppms_provideridentifier");

            if (detail != null && detail.Any())
            {
                _tracingService.Trace("---- Details: {0}", detail.Count());
            }
            else
            {
                _tracingService.Trace("---- No details found");
            }

            // Lookup validation setting
            _tracingService.Trace("Retrieve LEIE Exclusions");
            EntityCollection result = GetMatchingExclusions(npi);
            if (result != null && result.Entities.Count > 0)
            {
                _tracingService.Trace("Execute deactivate request - LEIE");
                _service.Execute(PpmsHelper.GetDeactivateRequest(provider, (int)PpmsHelper.Account_StatusCode.LeieExclusion));

                isDeactivateRequested = true;

                // Create batch detail result for validation failure
                CreateBatchDetailLeieResult(detail);
            }

            // Create NPI validation entry
            _tracingService.Trace("Create LEIE validation entries...");
            PpmsHelper.AddProviderValidation(_service, provider, PpmsHelper.Validation_Type.Leie);

            // Get PPMS service settings
            _serviceSettings = SettingsHelper.GetSettings(_service);
            if (!_serviceSettings.IsValid)
            {
                _tracingService.Trace("ERROR: Unable to retrieve service settings");
                return;
            }
            // 2.1 tweak for fortify; storage account key needs to be replaced
            _serviceSettings.StorageAccountKey = SecureStorageAccountKey;

            // Lookup NPPES status
            _tracingService.Trace("Check NPI Status: Npi{0} , NpiType{1}", npi, npiType);

            bool isActive = CheckNpiStatus(npi, npiType);
            if (!isActive)
            {
                if (!isDeactivateRequested)
                {
                    _tracingService.Trace("Execute deactivate request - NPI Status");
                    _service.Execute(PpmsHelper.GetDeactivateRequest(provider, (int)PpmsHelper.Account_StatusCode.NpiCheckFailure));
                }
                else
                {
                    _tracingService.Trace("Deactivation already requested - NPI Status");
                }

                // Create batch detail result for validation failure
                CreateBatchDetailNpiCheckResult(detail);
            }

            // Create NPI validation entry
            _tracingService.Trace("Create NPPES validation entries...");
            PpmsHelper.AddProviderValidation(_service, provider, PpmsHelper.Validation_Type.Nppes);
        }


        private void ValidateTin(Entity target)
        {
            // Get npi with provider reference
            var entity = GetNpi(target);
            if (entity == null)
            {
                _tracingService.Trace("Target entity not found.");
                return;
            }

            // Get associated provider reference
            _tracingService.Trace("Retrieve entity");
            var provider = entity.GetAttributeValue<EntityReference>("ppms_provider");
            // Get current batch detail
            var detail = GetCurrentBatchDetail(provider);
            var tin = entity.GetAttributeValue<string>("ppms_provideridentifier");

            // Lookup SAM.gov status
            _tracingService.Trace("Check SAM Status: TIN{0}", tin);

            bool isExcluded = GetExclusionStatus(tin, provider.Name);
            if (!isExcluded)
            {
                _tracingService.Trace("Execute deactivate request");
                _service.Execute(PpmsHelper.GetDeactivateRequest(provider, (int)PpmsHelper.Account_StatusCode.SamsExclusion));

                // Create batch detail result for validation failure
                CreateBatchDetailSamCheckResult(detail);
            }

            // Create SAM validation entry
            _tracingService.Trace("Create SAM validation entries...");
            PpmsHelper.AddProviderValidation(_service, provider, PpmsHelper.Validation_Type.Sam);
        }

        private Entity GetNpi(Entity entity)
        {
            if (entity != null)
            {
                return _service.Retrieve("ppms_provideridentifier", entity.Id, new ColumnSet(new string[] { "ppms_provideridentifier", "ppms_provider" }));
            }

            return null;
        }

        private EntityCollection GetMatchByNpi(string npi)
        {
            FilterExpression filter = new FilterExpression();
            filter.AddCondition("ppms_provideridentifier", ConditionOperator.Equal, npi.Trim());
            filter.AddCondition("statuscode", ConditionOperator.Equal, (int)PpmsHelper.Account_StatusCode.Active);

            QueryExpression query = new QueryExpression("ppms_provideridentifier");
            query.ColumnSet.AddColumns("ppms_provideridentifierid", "ppms_provideridentifier", "ppms_provider");
            query.Criteria.AddFilter(filter);

            return _service.RetrieveMultiple(query);
        }

        private EntityCollection GetMatchingExclusions(string npi)
        {
            FilterExpression filter = new FilterExpression();
            filter.AddCondition("ppms_npi", ConditionOperator.Equal, npi.Trim());
            filter.AddCondition("ppms_action", ConditionOperator.Equal, (int)PpmsHelper.LeieAction.Exclude);
            filter.AddCondition("statuscode", ConditionOperator.Equal, (int)PpmsHelper.Account_StatusCode.Active);

            QueryExpression query = new QueryExpression("ppms_leieexclusion");
            query.ColumnSet.AddColumns("ppms_npi", "ppms_action");
            query.Criteria.AddFilter(filter);

            return _service.RetrieveMultiple(query);
        }

        private void CreateBatchDetailLeieResult(IEnumerable<Entity> details)
        {
            if (details != null)
            {
                _tracingService.Trace("LEIE - Create batch detail result");
                foreach (var item in details)
                {
                    _tracingService.Trace("--- LEIE - {0}", item.Id);
                    CreateBatchDetailLeieResult(item.Id);
                }
            }
        }

        private void CreateBatchDetailLeieResult(Guid detailId)
        {
            // Add detail to batch
            var result = new Entity("ppms_batchdetailresult");
            result.Attributes.Add("ppms_name", "LEIE Exclusion");
            result.Attributes.Add("ppms_isvalid", false);
            result.Attributes.Add("ppms_entitytype", "Provider");
            result.Attributes.Add("ppms_result", "Affected by LEIE Exclusion rules");
            result.Attributes.Add("ppms_message", "The provider is on the current LEIE exclusion list.");
            result.Attributes.Add("ppms_batchdetail", new EntityReference("ppms_batchdetail", detailId));
            _service.Create(result);
        }

        private void CreateBatchDetailNpiCheckResult(IEnumerable<Entity> details)
        {
            if (details != null)
            {
                _tracingService.Trace("NPI Check - Create batch detail result");
                foreach (var item in details)
                {
                    _tracingService.Trace("--- NPI Check - {0}", item.Id);
                    CreateBatchDetailNpiCheckResult(item.Id);
                }
            }
        }

        private void CreateBatchDetailNpiCheckResult(Guid detailId)
        {
            // Add detail to batch
            var result = new Entity("ppms_batchdetailresult");
            result.Attributes.Add("ppms_name", "NPI Status Check");
            result.Attributes.Add("ppms_isvalid", false);
            result.Attributes.Add("ppms_entitytype", "Provider");
            result.Attributes.Add("ppms_result", "Affected by NPI status check rules");
            result.Attributes.Add("ppms_message", "The provider is not on the active list.");
            result.Attributes.Add("ppms_batchdetail", new EntityReference("ppms_batchdetail", detailId));
            _service.Create(result);
        }

        private void CreateBatchDetailSamCheckResult(IEnumerable<Entity> details)
        {
            if (details != null)
            {
                _tracingService.Trace("SAM - Create batch detail result");
                foreach (var item in details)
                {
                    _tracingService.Trace("--- SAM - {0}", item.Id);
                    CreateBatchDetailSamCheckResult(item.Id);
                }
            }
        }

        private void CreateBatchDetailSamCheckResult(Guid detailId)
        {
            // Add detail to batch
            var result = new Entity("ppms_batchdetailresult");
            result.Attributes.Add("ppms_name", "SAM Status Check");
            result.Attributes.Add("ppms_isvalid", false);
            result.Attributes.Add("ppms_entitytype", "Provider");
            result.Attributes.Add("ppms_result", "Affected by SAM status check rules");
            result.Attributes.Add("ppms_message", "The provider is not on the active list.");
            result.Attributes.Add("ppms_batchdetail", new EntityReference("ppms_batchdetail", detailId));
            _service.Create(result);
        }

        private Entity GetBatchDetail(Guid batchDetailId)
        {
            return _service.Retrieve("ppms_batchdetail", batchDetailId, new ColumnSet(new string[] { "ppms_batchdetailid", "ppms_name", "ppms_batch", ProcessDateAttribute }));
        }

        private Account GetProviderWithBatchDetail(Guid providerId)
        {
            using (var context = new PpmsContext(_service))
            {
                var provider = context.AccountSet.FirstOrDefault(a => a.AccountId == providerId);
                if (provider != null)
                {
                    context.LoadProperty(provider, new Relationship("ppms_account_batchdetail_provider"));
                }
                return provider;
            }
        }

        private IEnumerable<Entity> GetCurrentBatchDetail(EntityReference providerRef)
        {
            if (providerRef != null)
            {
                var provider = GetProviderWithBatchDetail(providerRef.Id);
                if (provider != null)
                {
                    // determine target batch detail
                    var children = provider.ppms_account_batchdetail_provider;
                    if (children != null && children.Any())
                    {
                        return provider.ppms_account_batchdetail_provider.OrderBy(o => o.CreatedOn).Where(a => a.ppms_validationcompletedate == null);
                    }
                }
            }

            return null;
        }

        private bool CheckNpiStatus(string npi, string npiType)
        {
            //The streamreader always returns a string even if there is no matching NPI 
            //so will do a check if Npi value is present. 
            var npiData = FabricHelper.GetNpiStatus(npi, npiType, _serviceSettings);

            _tracingService.Trace("Checking for Npi value {0} in NpiData: {1}", npi, npiData);
            if (string.IsNullOrEmpty(npiData))
            {
                _tracingService.Trace("NPI Data returned nothing: {0}", npiData);
                return true;
            }
            if (npiData.Contains(npi))
            {
                _tracingService.Trace("Macthing Npi value found: {0}", npiData);
                return true;
            }
            _tracingService.Trace("No Matching Npi value found {0}", npiData);
            return false;
        }

        /// <summary>
        /// Returns the exlcusion status for the provider based on SAM.gov information
        /// </summary>
        /// <param name="tin">Tax Identification Number</param>
        /// <param name="providerName">Name of the provider</param>
        /// <returns>true if found on the exclusion list, or if error occurs retrieving information</returns>
        private bool GetExclusionStatus(string tin, string providerName)
        {
            var op = new OperationExSSNSearchType();
            op.exactName = providerName;
            op.ssnOrTin = tin;

            // Search for exclusions
            var wf = new ExclusionSearchServiceBeanService();
            var response = wf.doSsnSearch(op);
            if (response.transactionInformation.successful)
            {
                if (response.excludedEntity != null)
                {
                    // Note exclusion types reported
                    foreach (var item in response.excludedEntity)
                    {
                        _tracingService.Trace($"Result: {item.exclusionType}");
                    }
                }
                else
                {
                    // No exclusions returned
                    _tracingService.Trace("Exclusion not found.");
                    return true;
                }
            }
            else
            {
                _tracingService.Trace("Unable to access Exclusion Search Service.");
            }

            return false;
        }
    }
}
